#!/usr/bin/env python3
"""
Quick test script to verify the OffersExtractor service is working
"""
import requests
import sys
from pathlib import Path

# Configuration
BASE_URL = "http://localhost:5100"  # Change to production URL as needed

def test_health():
    """Test the health endpoint"""
    print("🏥 Testing health endpoint...")
    try:
        response = requests.get(f"{BASE_URL}/health", timeout=5)
        print(f"Status Code: {response.status_code}")
        print(f"Response: {response.json()}")
        return response.status_code == 200
    except Exception as e:
        print(f"❌ Health check failed: {e}")
        return False

def test_extraction(pdf_path):
    """Test PDF extraction"""
    print(f"\n📄 Testing extraction with: {pdf_path}")
    
    if not Path(pdf_path).exists():
        print(f"❌ File not found: {pdf_path}")
        return False
    
    try:
        with open(pdf_path, 'rb') as f:
            files = {'pdf': f}
            response = requests.post(
                f"{BASE_URL}/extract_offers",
                files=files,
                timeout=300
            )
        
        print(f"Status Code: {response.status_code}")
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Extracted {data.get('count', 0)} offers")
            if data.get('offers'):
                print("\nFirst offer:")
                first_offer = data['offers'][0]
                for key, value in first_offer.items():
                    print(f"  {key}: {value}")
            return True
        else:
            print(f"❌ Extraction failed: {response.json()}")
            return False
            
    except Exception as e:
        print(f"❌ Extraction test failed: {e}")
        return False

def main():
    print("🧪 OffersExtractor Service Test\n")
    
    # Test health
    health_ok = test_health()
    
    if not health_ok:
        print("\n❌ Health check failed. Service may not be running.")
        sys.exit(1)
    
    # Test extraction if PDF path provided
    if len(sys.argv) > 1:
        pdf_path = sys.argv[1]
        extraction_ok = test_extraction(pdf_path)
        
        if not extraction_ok:
            sys.exit(1)
    else:
        print("\n💡 To test extraction, run:")
        print(f"   python test_service.py /path/to/test.pdf")
    
    print("\n✅ All tests passed!")

if __name__ == "__main__":
    main()
